﻿using DataAccess.DataRows;
using DataAccess.DataTables;
using DataAccess.TableAdapters;
using Microsoft.DirectX.AudioVideoPlayback;
using System;
using System.Collections;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Drawing;
using System.Net.Sockets;
using System.Threading;
using System.Windows.Forms;

public class Graphics
{
    private bool _endOfProgram = false;
    private const int WIDTH = 600, HEIGHT = 600;
    private CurrentStateDataRow _currentState;
    private Form _mainForm;
    private MenuItem _inputMI, _textBoxMI, _physicianNumberMI, _introScriptMI, _nameMI;
    private MessageDataRow _currentMessage;
    private Panel _manualInputPanel, _headPanel;
    private ServerClient _client;
    private string _patientName = "";
    private TextBox _textBox;
    private Process _process;
    private Video _video;

    public Graphics()
    {
        this.InitializeComponent();

        Application.Run(_mainForm);
    }

    #region Initialize

    private void InitializeComponent()
    {
        this.createMainForm();

        this.createPanels();

        this.createTextBox();

        this.createMenusAndAddComponentsToForm();

        this.killOtherXfaceProcesses();

        this.loadAndPlayIntroVideo();
    }

    private void createMainForm()
    {
        _mainForm = new Form();
        _mainForm.BackColor = Color.White;
        _mainForm.Text = "Talking Head";
        _mainForm.MinimumSize = new Size(WIDTH, HEIGHT);
        _mainForm.StartPosition = FormStartPosition.CenterScreen;
        _mainForm.FormClosing += new FormClosingEventHandler(this.formClosing);
    }

    private void createPanels()
    {
        _manualInputPanel = new Panel();
        _manualInputPanel.Size = new Size((int)(_mainForm.Width * .3), (int)(_mainForm.Height * .8));
        _manualInputPanel.BackColor = Color.White;
        _manualInputPanel.BorderStyle = BorderStyle.Fixed3D;
        _manualInputPanel.Dock = DockStyle.Right;

        _headPanel = new Panel();
        _headPanel.Size = new Size((int)(_mainForm.Width * .8), (int)(_mainForm.Height * .8));
        _headPanel.BackColor = Color.White;
        _headPanel.BorderStyle = BorderStyle.Fixed3D;
        _headPanel.Dock = DockStyle.Fill;
        _headPanel.SizeChanged += new EventHandler(this.headPanelSizeChanged);
    }

    private void createTextBox()
    {
        _textBox = new TextBox();
        _textBox.Size = new Size(_mainForm.Width, (int)(_mainForm.Height * .2));
        _textBox.Font = new Font("Times New Roman", 20);
        _textBox.ReadOnly = true;
        _textBox.BackColor = Color.White;
        _textBox.BorderStyle = BorderStyle.Fixed3D;
        _textBox.ForeColor = Color.Black;
        _textBox.Multiline = true;
        _textBox.ScrollBars = ScrollBars.Vertical;
        _textBox.Dock = DockStyle.Bottom;
        _textBox.GotFocus += new EventHandler(this.textBoxGetsFocus);
    }

    private void createMenusAndAddComponentsToForm()
    {
        MenuItem exportMI = new MenuItem("Export to Excel");
        exportMI.Name = "EXPORTMI";
        exportMI.Click += new EventHandler(this.menuItemClicked);

        MenuItem exitMI = new MenuItem("Exit");
        exitMI.Name = "EXITMI";
        exitMI.Click += new EventHandler(this.menuItemClicked);

        MenuItem fileMenu = new MenuItem("File");
        fileMenu.MenuItems.AddRange(new MenuItem[] { exportMI, exitMI });

        _textBoxMI = new MenuItem("Hide Captions");
        _textBoxMI.Click += new EventHandler(this.menuItemClicked);

        _inputMI = new MenuItem("Hide Input");
        _inputMI.Click += new EventHandler(this.menuItemClicked);

        MenuItem viewMenu = new MenuItem("View");
        viewMenu.MenuItems.AddRange(new MenuItem[] { _inputMI, _textBoxMI });

        _introScriptMI = new MenuItem("Play Intro Script");
        _introScriptMI.Click += new EventHandler(this.menuItemClicked);

        _physicianNumberMI = new MenuItem("Change Physician's #");
        _physicianNumberMI.Click += new EventHandler(this.menuItemClicked);

        _nameMI = new MenuItem("Change Name");
        _nameMI.Click += new EventHandler(this.menuItemClicked);

        MenuItem configMenu = new MenuItem("Configuration");
        configMenu.MenuItems.AddRange(new MenuItem[] { _introScriptMI, _nameMI, _physicianNumberMI });

        MainMenu mainMenu = new MainMenu();
        mainMenu.MenuItems.AddRange(new MenuItem[] { fileMenu, viewMenu, configMenu });

        _mainForm.Menu = mainMenu;
        _mainForm.Controls.Add(_headPanel);
        _mainForm.Controls.Add(_manualInputPanel);
        _mainForm.Controls.Add(_textBox);

        _mainForm.Show();
    }

    private void killOtherXfaceProcesses()
    {
        try
        {
            for (int x = 0; x < Process.GetProcesses().Length; x++)
            {
                Process killProcess = Process.GetProcesses()[x];
                if (killProcess.ProcessName.CompareTo("XfacePlayer") == 0)
                {
                    killProcess.Kill();
                }
            }
        }
        catch
        {}
    }

    private void loadAndPlayIntroVideo()
    {
        _video = new Video("Alice Intro.avi");
        _video.Owner = _headPanel;
        _video.Ending += new EventHandler(this.videoEnding);

        _video.Play();
    }

    private void startXfacePlayer()
    {
        _process = Process.Start("XfacePlayer.exe");
        _process.WaitForInputIdle();

        SetParent(_process.MainWindowHandle, _headPanel.Handle);
        _process.WaitForInputIdle();

        MoveWindow(_process.MainWindowHandle, 0, 0, _headPanel.Width, _headPanel.Height, true);
        _process.WaitForInputIdle();
    }

    #endregion

    #region ClosingForm

    private void formClosing(object sender, FormClosingEventArgs eventArgs)
    {
        eventArgs.Cancel = true;
        this.closeForm();
    }

    private void closeForm()
    {
        _video.Dispose();

        if (_endOfProgram == false)
        {
            try
            {
                new CurrentStateTableAdapter().InsertCurrentState(1, 0, _currentMessage.ID);
            }
            catch
            { }
        }

        try
        {
            _client.stopPlayBack();
            _client.close();
        }
        catch
        { }
        try
        {
            _process.CloseMainWindow();
            _process.Kill();
        }
        catch
        { }
        System.Environment.Exit(0);
    }

    #endregion

    #region ScriptEvents
    
    public void makeHeadTalk(string script, int groupEntry)
    {
        var thread = new Thread(() => _client.sendAndPlayScript(script, groupEntry));
        thread.Start();
    }

    private void checkState()
    {
        try
        {
            _patientName = new ConfigurationTableAdapter().FillByKey("PATIENT_NAME").Value;

            _currentState = new CurrentStateTableAdapter().FillByAll();

            //if this is the first time running the program, the intro script is the first thing to play
            if (_currentState.RunStartupScript == 1)
            {
                GroupControlDataRow introScriptGroup = new GroupControlTableAdapter().FillByEntry(6)[0];
                MessageDataTable introMessage = new MessageTableAdapter().FillByGroupID(introScriptGroup.ID);

                int random = new Random().Next(introMessage.Rows.Count);

                _currentMessage = ((MessageDataRow)introMessage.Rows[random]);
                
                _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

                _mainForm.Refresh();

                _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), 6);
                return;
            }

            //if the program wasn't currently in progress
            if (_currentState.InProgress == 0)
            {
                GroupControlDataRow startingGroup = new GroupControlTableAdapter().FillByID(1);
                MessageDataTable startingMessages = new MessageTableAdapter().FillByGroupID(startingGroup.ID);

                int random = new Random().Next(startingMessages.Rows.Count);

                _currentMessage = ((MessageDataRow)startingMessages.Rows[random]);
                _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

                _mainForm.Refresh();
                _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), 0);
            }
            //if the program was currently in progress
            else
            {
                _currentMessage = new MessageTableAdapter().FillByID(_currentState.CurrentMessageID);

                this.loadContinuedMessage();
            }
        }
        catch
        {
            _process.Kill();
            System.Environment.Exit(0);
        }
    }

    public void loadNextScene()
    {
        _manualInputPanel.Controls.Clear();
        MessageLinkDataRow nextQuestionLink = new MessageLinkTableAdapter().FillByParent(_currentMessage.ID);

        GroupControlDataRow nextGroup = new GroupControlTableAdapter().FillByID(nextQuestionLink.Child);
        MessageDataTable nextMessages = new MessageTableAdapter().FillByGroupID(nextGroup.ID);

        new CurrentStateTableAdapter().InsertCurrentState(1, 0, _currentMessage.ID);

        //if the group is a group of statements/questions
        if (nextGroup.Entry == 0 || nextGroup.Entry == 10)
        {
            int random = new Random().Next(nextMessages.Rows.Count);
            _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), nextGroup.Entry);
        }
        //if the group is a group of statements that needs a continue button
        else if (nextGroup.Entry == 1)
        {
            int random = new Random().Next(nextMessages.Rows.Count);
            _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), nextGroup.Entry);
        }
        //if the group is a group for glucose readings
        else if (nextGroup.Entry == 2)
        {
            this.setupGlucoseReading(nextMessages);
        }
        //if the group is a group for pedometer readings
        else if (nextGroup.Entry == 3)
        {
            int random = new Random().Next(nextMessages.Rows.Count);
            _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), nextGroup.Entry);
        }
        //if the group is a group for user reponse
        else if (nextGroup.Entry == 4)
        {
            this.addUserResponse(nextMessages);
        }
        //if the group is a group for ending the conversation
        else if (nextGroup.Entry == 5)
        {
            int random = new Random().Next(nextMessages.Rows.Count);
            _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            new CurrentStateTableAdapter().InsertCurrentState(0, 0, _currentMessage.ID);
            _endOfProgram = true;

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), nextGroup.Entry);
        }
        //doctor number input
        else if (nextGroup.Entry == 11)
        {
            int random = new Random().Next(nextMessages.Rows.Count);
            _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), nextGroup.Entry);
        }
    }

    public void loadContinuedMessage()
    {
        _manualInputPanel.Controls.Clear();

        GroupControlDataRow currentGroup = new GroupControlTableAdapter().FillByID(_currentMessage.GroupID);

        new CurrentStateTableAdapter().InsertCurrentState(1, 0, _currentMessage.ID);

        //if the group is a group of statements/questions
        if (currentGroup.Entry == 0 || currentGroup.Entry == 8 || currentGroup.Entry == 9 || currentGroup.Entry == 10)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), currentGroup.Entry);
        }
        //if the group is a group of statements that needs a continue button
        else if (currentGroup.Entry == 1)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName).Replace("(PHYSICIANNUMBER)", new ConfigurationTableAdapter().FillByKey("PHYSICIAN_NUMBER").Value);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName).Replace("(PHYSICIANNUMBER)", new ConfigurationTableAdapter().FillByKey("PHYSICIAN_NUMBER").Value), currentGroup.Entry);
        }
        //if the group is a group for glucose readings
        else if (currentGroup.Entry == 2)
        {
            MessageDataTable nextMessages = new MessageTableAdapter().FillByGroupID(currentGroup.ID);
            this.setupGlucoseReading(nextMessages);
        }
        //if the group is a group for pedometer readings
        else if (currentGroup.Entry == 3)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), currentGroup.Entry);
        }
        //if the group is a group for user reponse
        else if (currentGroup.Entry == 4)
        {
            MessageDataTable nextMessages = new MessageTableAdapter().FillByGroupID(currentGroup.ID);
            this.addUserResponse(nextMessages);
        }
        //if the group is a group for ending the conversation
        else if (currentGroup.Entry == 5)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            new CurrentStateTableAdapter().InsertCurrentState(0, 0, _currentMessage.ID);
            _endOfProgram = true;

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), currentGroup.Entry);
        }
        //if the group is a group for name prompt
        else if (currentGroup.Entry == 6)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), currentGroup.Entry);
        }
        //doctor number input
        else if (currentGroup.Entry == 11)
        {
            _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

            _mainForm.Refresh();
            _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), currentGroup.Entry);
        }
    }

    #endregion

    #region SetupForScriptEvents

    public void setupPedometerReading()
    {
        TextBox pedoMeterInput = new TextBox();
        pedoMeterInput.Size = new Size(_manualInputPanel.Width, 100);
        pedoMeterInput.Location = new Point(0, 0);
        pedoMeterInput.Name = "PEDOMETER_INPUT";
        pedoMeterInput.TextAlign = HorizontalAlignment.Center;
        pedoMeterInput.Font = new Font("Times New Roman", 15);
        pedoMeterInput.ReadOnly = false;
        pedoMeterInput.BackColor = Color.White;
        pedoMeterInput.ForeColor = Color.Black;
        pedoMeterInput.Multiline = false;

        Button submitButton = new Button();
        submitButton.Text = "Submit";
        submitButton.Size = new Size((int)(_manualInputPanel.Width * .75), 60);
        submitButton.BackColor = Color.White;
        submitButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), pedoMeterInput.Height + 20);
        submitButton.Click += new EventHandler(this.pedoMeterEnteredButtonEvent);

        _manualInputPanel.Controls.Add(pedoMeterInput);
        _manualInputPanel.Controls.Add(submitButton);
    }

    public void setupGlucoseReading(MessageDataTable glucoseMessages)
    {
        MessageDataRow previousMessage = ((MessageDataRow)glucoseMessages.Rows[0]);
        _currentMessage = ((MessageDataRow)glucoseMessages.Rows[1]);

        int height = 0;

        TextBox glucoseTimeMessage = new TextBox();
        glucoseTimeMessage.Size = new Size(_manualInputPanel.Width, 100);
        glucoseTimeMessage.Location = new Point(0, height);
        glucoseTimeMessage.Text = previousMessage.Text.Replace("(PATIENT)", _patientName);
        glucoseTimeMessage.Name = "GLUCOSE_TIME_MESSAGE";
        glucoseTimeMessage.Font = new Font("Times New Roman", 15);
        glucoseTimeMessage.ReadOnly = true;
        glucoseTimeMessage.BackColor = Color.White;
        glucoseTimeMessage.ForeColor = Color.Black;
        glucoseTimeMessage.Multiline = true;

        height += glucoseTimeMessage.Height;

        DateTimePicker glucoseTimeSelect = new DateTimePicker();
        glucoseTimeSelect.Size = new Size((int)(_manualInputPanel.Width + 20), 60);
        glucoseTimeSelect.Location = new Point(0, height);
        glucoseTimeSelect.Format = DateTimePickerFormat.Time;
        glucoseTimeSelect.Font = new Font("Times New Roman", 15);

        height += glucoseTimeSelect.Height;

        TextBox glucoseNumberMessage = new TextBox();
        glucoseNumberMessage.Size = new Size(_manualInputPanel.Width, 100);
        glucoseNumberMessage.Location = new Point(0, height);
        glucoseNumberMessage.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);
        glucoseNumberMessage.Name = "GLUCOSE_NUMBER_MESSAGE";
        glucoseNumberMessage.Font = new Font("Times New Roman", 15);
        glucoseNumberMessage.ReadOnly = true;
        glucoseNumberMessage.BackColor = Color.White;
        glucoseNumberMessage.ForeColor = Color.Black;
        glucoseNumberMessage.Multiline = true;

        height += glucoseNumberMessage.Height;

        TextBox glucoseNumberInput = new TextBox();
        glucoseNumberInput.Size = new Size(_manualInputPanel.Width, 100);
        glucoseNumberInput.Location = new Point(0, height);
        glucoseNumberInput.Name = "GLUCOSE_NUMBER_INPUT";
        glucoseNumberInput.TextAlign = HorizontalAlignment.Center;
        glucoseNumberInput.Font = new Font("Times New Roman", 15);
        glucoseNumberInput.ReadOnly = false;
        glucoseNumberInput.BackColor = Color.White;
        glucoseNumberInput.ForeColor = Color.Black;
        glucoseNumberInput.Multiline = false;

        height += glucoseNumberInput.Height;

        Button submitButton = new Button();
        submitButton.Text = "Submit";
        submitButton.Name = previousMessage.ID.ToString();
        submitButton.Size = new Size((int)(_manualInputPanel.Width * .75), 60);
        submitButton.BackColor = Color.White;
        submitButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), height + 20);
        submitButton.Click += new EventHandler(this.glucoseEnteredButtonEvent);

        _manualInputPanel.Controls.Add(glucoseTimeMessage);
        _manualInputPanel.Controls.Add(glucoseTimeSelect);
        _manualInputPanel.Controls.Add(glucoseNumberMessage);
        _manualInputPanel.Controls.Add(glucoseNumberInput);
        _manualInputPanel.Controls.Add(submitButton);
    }

    public void setupPhysiciansNumber(string name)
    {
        TextBox physicianNumber = new TextBox();
        physicianNumber.Size = new Size(_manualInputPanel.Width, 100);
        physicianNumber.Location = new Point(0, 0);
        physicianNumber.Name = "PHYSICIAN_NUMBER_INPUT";
        physicianNumber.TextAlign = HorizontalAlignment.Center;
        physicianNumber.Font = new Font("Times New Roman", 15);
        physicianNumber.ReadOnly = false;
        physicianNumber.BackColor = Color.White;
        physicianNumber.ForeColor = Color.Black;
        physicianNumber.Multiline = false;

        Button submitButton = new Button();
        submitButton.Text = "Submit";
        submitButton.Name = name;
        submitButton.Size = new Size((int)(_manualInputPanel.Width * .75), 60);
        submitButton.BackColor = Color.White;
        submitButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), physicianNumber.Height + 20);
        submitButton.Click += new EventHandler(this.physicianNumberEnteredButtonEvent);

        _manualInputPanel.Controls.Add(physicianNumber);
        _manualInputPanel.Controls.Add(submitButton);
    }

    public void addUserResponse(MessageDataTable messageDataTable)
    {
        int locX = 5;
        int locY = 0;

        for (int x = 0; x < messageDataTable.Rows.Count; x++)
        {
            RadioButton button = new RadioButton();
            button.Text = ((MessageDataRow)messageDataTable.Rows[x]).Text;
            button.Name = ((MessageDataRow)messageDataTable.Rows[x]).ID.ToString();
            button.Size = new Size(_manualInputPanel.Width, ((int)((double)(_manualInputPanel.Height - 75) / messageDataTable.Rows.Count)));
            button.Font = new Font("Times New Roman", 12);
            button.BackColor = Color.White;
            button.Location = new Point(locX, locY);

            _manualInputPanel.Controls.Add(button);
            locY += button.Height;
        }

        Button submitButton = new Button();
        submitButton.Text = "Submit";
        submitButton.Name = "";
        submitButton.Size = new Size((int)(_manualInputPanel.Width*.6), 50);
        submitButton.BackColor = Color.White;
        submitButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), locY);
        submitButton.Click += new EventHandler(this.submitButtonEvent);

        _manualInputPanel.Controls.Add(submitButton);
    }

    public void addContinueButton()
    {
        Button continueButton = new Button();
        continueButton.Text = "Continue";
        continueButton.Name = _currentMessage.ID.ToString();
        continueButton.Size = new Size((int)(_manualInputPanel.Width * .75), 100);
        continueButton.BackColor = Color.White;
        continueButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), (int)(_manualInputPanel.Height / 4.0));
        continueButton.Click += new EventHandler(this.continueButtonEvent);

        _manualInputPanel.Controls.Add(continueButton);
    }

    public void setupNamePrompt(string name)
    {
        TextBox nameBox = new TextBox();
        nameBox.Size = new Size(_manualInputPanel.Width, 100);
        nameBox.Location = new Point(0, 0);
        nameBox.Name = "NAME_INPUT";
        nameBox.TextAlign = HorizontalAlignment.Center;
        nameBox.Font = new Font("Times New Roman", 15);
        nameBox.ReadOnly = false;
        nameBox.BackColor = Color.White;
        nameBox.ForeColor = Color.Black;
        nameBox.Multiline = false;

        Button submitButton = new Button();
        submitButton.Text = "Submit";
        submitButton.Name = name;
        submitButton.Size = new Size((int)(_manualInputPanel.Width * .75), 60);
        submitButton.BackColor = Color.White;
        submitButton.Location = new Point((int)(_manualInputPanel.Width / 8.0), nameBox.Height + 20);
        submitButton.Click += new EventHandler(this.nameEnteredButtonEvent);

        _manualInputPanel.Controls.Add(nameBox);
        _manualInputPanel.Controls.Add(submitButton);
    }

    #endregion

    #region IntroScript

    public void playIntroScript()
    {
        _manualInputPanel.Controls.Clear();

        GroupControlDataTable introScriptGroup = new GroupControlTableAdapter().FillByEntry(10);
        MessageDataTable introScript = new MessageTableAdapter().FillByGroupID(((GroupControlDataRow)introScriptGroup[0]).ID);

        MessageDataRow currentMessage = ((MessageDataRow)introScript.Rows[0]);
        _textBox.Text = currentMessage.Text.Replace("(PATIENT)", _patientName);

        _mainForm.Refresh();

        _currentMessage = new MessageTableAdapter().FillByID(new CurrentStateTableAdapter().FillByAll().CurrentMessageID);

        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), currentMessage.Script.Replace("(PATIENT)", _patientName), new GroupControlTableAdapter().FillByID(currentMessage.GroupID).Entry);
    }

    public void continueAfterIntroScript()
    {
        _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

        _mainForm.Refresh();

        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), new GroupControlTableAdapter().FillByID(_currentMessage.GroupID).Entry);
    }

    #endregion

    #region PhysicianNumberScript

    public void askForPhysicianNumber()
    {
        _manualInputPanel.Controls.Clear();

        GroupControlDataTable doctorScriptGroup = new GroupControlTableAdapter().FillByEntry(11);
        MessageDataTable doctorScript = new MessageTableAdapter().FillByGroupID(((GroupControlDataRow)doctorScriptGroup[0]).ID);

        MessageDataRow currentMessage = ((MessageDataRow)doctorScript.Rows[0]);
        _textBox.Text = currentMessage.Text.Replace("(PATIENT)", _patientName);

        _mainForm.Refresh();

        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), currentMessage.Script.Replace("(PATIENT)", _patientName), -1);
    }

    #endregion

    #region NameScript

    public void askForNewName()
    {
        _manualInputPanel.Controls.Clear();

        GroupControlDataTable nameScriptGroup = new GroupControlTableAdapter().FillByEntry(6);
        MessageDataTable nameScript = new MessageTableAdapter().FillByGroupID(((GroupControlDataRow)nameScriptGroup[0]).ID);

        MessageDataRow currentMessage = ((MessageDataRow)nameScript.Rows[0]);
        _textBox.Text = currentMessage.Text.Replace("(PATIENT)", _patientName);

        _mainForm.Refresh();

        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), currentMessage.Script.Replace("(PATIENT)", _patientName), -2);
    }

    #endregion

    #region ResponseEvents

    public void submitButtonEvent(object sender, EventArgs e)
    {
        for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
        {
            if (_manualInputPanel.Controls[x].GetType() == typeof(RadioButton))
            {
                RadioButton radioButton = (RadioButton)_manualInputPanel.Controls[x];
                if (radioButton.Checked == true)
                {
                    new ResponseHistoryTableAdapter().InsertResponse(radioButton.Text);

                    int messageID = Convert.ToInt32(radioButton.Name);
                    _currentMessage = new MessageTableAdapter().FillByID(messageID);

                    this.loadNextScene();
                }
            }
        }
    }

    public void continueButtonEvent(object sender, EventArgs e)
    {
        Button continueButton = (Button)sender;
        int messageID = Convert.ToInt32(continueButton.Name);
        _currentMessage = new MessageTableAdapter().FillByID(messageID);

        this.loadNextScene();
    }

    public void glucoseEnteredButtonEvent(object sender, EventArgs e)
    {
        TextBox glucoseNumberInput = null;
        DateTimePicker glucoseTimeInput = null;

        for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
        {
            if (_manualInputPanel.Controls[x].GetType() == typeof(TextBox) && ((TextBox)_manualInputPanel.Controls[x]).Name.CompareTo("GLUCOSE_NUMBER_INPUT") == 0)
            {
                glucoseNumberInput = (TextBox)_manualInputPanel.Controls[x];
            }
            else if (_manualInputPanel.Controls[x].GetType() == typeof(DateTimePicker))
            {
                glucoseTimeInput = (DateTimePicker)_manualInputPanel.Controls[x];
            }
        }

        int glucoseNumber;
        if (int.TryParse(glucoseNumberInput.Text, out glucoseNumber) == true)
        {
            string day = "", month = "", year = "", date = "", hour = "", minute = "", time = "";

            if (glucoseTimeInput.Value.Day < 10)
            {
                day = "0";
            }
            day += glucoseTimeInput.Value.Day.ToString();

            if (glucoseTimeInput.Value.Month < 10)
            {
                month = "0";
            }
            month += glucoseTimeInput.Value.Month.ToString();
            year = glucoseTimeInput.Value.Year.ToString();
            date = (year + "-" + month + "-" + day);

            if (glucoseTimeInput.Value.Hour < 10)
            {
                hour = "0";
            }
            hour += glucoseTimeInput.Value.Hour.ToString();

            if (glucoseTimeInput.Value.Minute < 10)
            {
                minute = "0";
            }
            minute += glucoseTimeInput.Value.Minute.ToString();
            time = (hour + ":" + minute);

            Console.WriteLine(time);

            bool value = new GlucoseTableAdapter().InsertGlucose(glucoseNumber, date, time);

            //if the last glucose reading was completed
            if ((new MessageLinkTableAdapter().FillByParent(_currentMessage.ID)).Child != new MessageLinkTableAdapter().FillByParent(Int32.Parse(((Button)sender).Name)).Child)
            {
                GlucoseDataTable table = new GlucoseTableAdapter().FillByAll();
                string times = "";

                for (int x = (table.Rows.Count - 1); x > (table.Rows.Count - 5); x--)
                {
                    if (table[x].GlucoseLevel < 100)
                    {
                        times += table[x].Time + " ";
                    }
                }

                //if there are glucose readings that are off
                if (times.CompareTo("") != 0)
                {
                    _currentMessage = new MessageTableAdapter().FillByID(Int32.Parse(((Button)sender).Name));
                }

                _manualInputPanel.Controls.Clear();
                MessageLinkDataRow nextQuestionLink = new MessageLinkTableAdapter().FillByParent(_currentMessage.ID);

                GroupControlDataRow nextGroup = new GroupControlTableAdapter().FillByID(nextQuestionLink.Child);
                MessageDataTable nextMessages = new MessageTableAdapter().FillByGroupID(nextGroup.ID);

                new CurrentStateTableAdapter().InsertCurrentState(1, 0, _currentMessage.ID);

                int random = new Random().Next(nextMessages.Rows.Count);
                _currentMessage = ((MessageDataRow)nextMessages.Rows[random]);
                _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName).Replace("(TIMES)", times);

                _mainForm.Refresh();

                _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName).Replace("(TIMES)", times), new GroupControlTableAdapter().FillByID(_currentMessage.GroupID).Entry);
            }
            //if the last glucose reading is not completed
            else
            {
                this.loadNextScene();
            }
        }
        else
        {
            glucoseNumberInput.Text = string.Empty;
        }
    }

    public void pedoMeterEnteredButtonEvent(object sender, EventArgs e)
    {
        for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
        {
            if (_manualInputPanel.Controls[x].GetType() == typeof(TextBox) && ((TextBox)_manualInputPanel.Controls[x]).Name.CompareTo("PEDOMETER_INPUT") == 0)
            {
                int pedometerNumber;
                if (int.TryParse(((TextBox)_manualInputPanel.Controls[x]).Text, out pedometerNumber))
                {
                    new ExerciseTableAdapter().InsertExercise(pedometerNumber);
                    
                    GroupControlDataRow nextGroup;
                    if (pedometerNumber < 10000)
                    {
                        nextGroup = new GroupControlTableAdapter().FillByEntry(9)[0];
                    }
                    else
                    {
                        nextGroup = new GroupControlTableAdapter().FillByEntry(8)[0];
                    }

                    MessageDataTable pedometerResponses = new MessageTableAdapter().FillByGroupID(nextGroup.ID);

                    int random = new Random().Next(pedometerResponses.Rows.Count);

                    _currentMessage = ((MessageDataRow)pedometerResponses.Rows[random]);

                    this.loadContinuedMessage();
                }
                else
                {
                    ((TextBox)_manualInputPanel.Controls[x]).Text = string.Empty;
                }
                break;
            }
        }
    }

    public void physicianNumberEnteredButtonEvent(object sender, EventArgs e)
    {
        for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
        {
            if (_manualInputPanel.Controls[x].GetType() == typeof(TextBox))
            {
                string physiciansNumber = ((TextBox)_manualInputPanel.Controls[x]).Text;
                if (physiciansNumber != null)
                {
                    new ConfigurationTableAdapter().UpdateConfigurationByKey("PHYSICIAN_NUMBER", physiciansNumber);
                    if (((Button)sender).Name.CompareTo("NEXTSCENE") == 0)
                    {
                        _currentMessage = new MessageTableAdapter().FillByID(new CurrentStateTableAdapter().FillByAll().CurrentMessageID);
                        _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

                        _mainForm.Refresh();

                        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), new GroupControlTableAdapter().FillByID(_currentMessage.GroupID).Entry);
                        return;
                    }

                    this.loadNextScene();
                }
            }
        }
    }

    public void nameEnteredButtonEvent(object sender, EventArgs e)
    {
        for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
        {
            if (_manualInputPanel.Controls[x].GetType() == typeof(TextBox))
            {
                string name = ((TextBox)_manualInputPanel.Controls[x]).Text;
                if (name != null)
                {
                    new ConfigurationTableAdapter().UpdateConfigurationByKey("PATIENT_NAME", name);
                    _patientName = name;
                    if (((Button)sender).Name.CompareTo("NEXTSCENE") == 0)
                    {
                        _currentMessage = new MessageTableAdapter().FillByID(new CurrentStateTableAdapter().FillByAll().CurrentMessageID);
                        _textBox.Text = _currentMessage.Text.Replace("(PATIENT)", _patientName);

                        _mainForm.Refresh();

                        _mainForm.Invoke(new Graphics.makeHeadTalkDelegate(this.makeHeadTalk), _currentMessage.Script.Replace("(PATIENT)", _patientName), new GroupControlTableAdapter().FillByID(_currentMessage.GroupID).Entry);
                        return;
                    }

                    this.loadNextScene();
                }
            }
        }
    }

    public void loadAndPlayEndingVideo()
    {
        _video.Dispose();

        try
        {
            _client.stopPlayBack();
            _client.close();
        }
        catch
        { }
        try
        {
            _process.CloseMainWindow();
            _process.Kill();
        }
        catch
        { }

        _mainForm.Refresh();

        _video = new Video("Alice End.avi");
        _video.Owner = _headPanel;
        _video.Ending += new EventHandler(this.videoEnding);

        _video.Play();
    }

    public void videoEnding(object sender, EventArgs e)
    {
        if (_endOfProgram == false)
        {
            this.startXfacePlayer();

            _client = new ServerClient(this);
            _client.connect();

            this.checkState();
        }
        else
        {
            _physicianNumberMI.Enabled = false;
            _introScriptMI.Enabled = false;
            _nameMI.Enabled = false;

            _video.Size = new Size(0, 0);
        }
    }

    #endregion

    #region MenuItemEvents

    public void menuItemClicked(object sender, EventArgs eventArgs)
    {
        //if _textBoxMI is clicked
        if (sender.Equals(_textBoxMI))
        {
            if (_textBoxMI.Text.CompareTo("Show Captions") == 0)
            {
                _textBoxMI.Text = "Hide Captions";
                _textBox.Visible = true;
            }
            else
            {
                _textBoxMI.Text = "Show Captions";
                _textBox.Visible = false;
            }
        }
        //if _inputMI is clicked
        else if (sender.Equals(_inputMI))
        {
            if (_inputMI.Text.CompareTo("Show Input") == 0)
            {
                _inputMI.Text = "Hide Input";
                _manualInputPanel.Visible = true;
                for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
                {
                    _manualInputPanel.Controls[x].Visible = true;
                }
            }
            else
            {
                _inputMI.Text = "Show Input";
                for (int x = 0; x < _manualInputPanel.Controls.Count; x++)
                {
                    _manualInputPanel.Controls[x].Visible = false;
                }
                _manualInputPanel.Visible = false;
            }
        }
        //if exitMenuItem is clicked
        else if (((MenuItem)sender).Name.CompareTo("EXITMI") == 0)
        {
            this.closeForm();
        }
        //if the user wants to play the intro script again
        else if (sender.Equals(_introScriptMI))
        {
            this.playIntroScript();
        }
        //if _phoneNumberMI is clicked
        else if (sender.Equals(_physicianNumberMI))
        {
            this.askForPhysicianNumber();
        }
        //if _nameMI is clicked
        else if (sender.Equals(_nameMI))
        {
            this.askForNewName();
        }
        //if exportMI is clicked
        else if (((MenuItem)sender).Name.CompareTo("EXPORTMI") == 0)
        {
            this.exportToExcel();
        }
    }

    public void exportToExcel()
    {
        Microsoft.Office.Interop.Excel.Application excelApp = new Microsoft.Office.Interop.Excel.Application();
        Microsoft.Office.Interop.Excel.Workbook workBook = excelApp.Workbooks.Add(Microsoft.Office.Interop.Excel.XlSheetType.xlWorksheet);
        Microsoft.Office.Interop.Excel.Worksheet workSheet = (Microsoft.Office.Interop.Excel.Worksheet)excelApp.ActiveSheet;
        workSheet.Name = "Database Reports";

        Microsoft.Office.Interop.Excel.Range range = workSheet.get_Range("A1", "C1");
        range.Select();
        range.Font.Bold = true;
        range.Font.Name = "Arial";
        range.Font.Size = 12;
        range.MergeCells = true;
        range.HorizontalAlignment = Microsoft.Office.Interop.Excel.Constants.xlCenter;
        range.Borders.Weight = 1;
        range.Borders.LineStyle = Microsoft.Office.Interop.Excel.Constants.xlSolid;

        workSheet.Cells[1, 1] = "Glucose";

        range = workSheet.get_Range("E1", "F1");
        range.Select();
        range.Font.Bold = true;
        range.Font.Name = "Arial";
        range.Font.Size = 12;
        range.MergeCells = true;
        range.HorizontalAlignment = Microsoft.Office.Interop.Excel.Constants.xlCenter;
        range.Borders.Weight = 1;
        range.Borders.LineStyle = Microsoft.Office.Interop.Excel.Constants.xlSolid;

        workSheet.Cells[1, 5] = "Pedometer";

        range = workSheet.get_Range("A2", "F2");
        range.Select();
        range.Font.Bold = true;
        range.Font.Name = "Arial";
        range.Font.Size = 10;
        range.ColumnWidth = 18;
        range.HorizontalAlignment = Microsoft.Office.Interop.Excel.Constants.xlCenter;
        range.Borders.Weight = 1;
        range.Borders.LineStyle = Microsoft.Office.Interop.Excel.Constants.xlSolid;

        workSheet.Cells[2, 1] = "Date";
        workSheet.Cells[2, 2] = "Time";
        workSheet.Cells[2, 3] = "Glucose Level";
        workSheet.Cells[2, 5] = "Date";
        workSheet.Cells[2, 6] = "Total Steps";

        int row = 3;
        GlucoseDataTable glucoseDataTable = new GlucoseTableAdapter().FillByAll();
        for (int x = glucoseDataTable.Rows.Count - 1; x >= 0; x--)
        {
            range = workSheet.get_Range("A" + (row).ToString(), "C" + (row).ToString());
            range.Select();
            range.Font.Name = "Arial";
            range.HorizontalAlignment = Microsoft.Office.Interop.Excel.Constants.xlCenter;

            GlucoseDataRow glucoseDataRow = (GlucoseDataRow)glucoseDataTable.Rows[x];

            workSheet.Cells[row, 1] = glucoseDataRow.Date.Substring(0, 9);
            workSheet.Cells[row, 2] = glucoseDataRow.Time.Substring(9);
            workSheet.Cells[row, 3] = glucoseDataRow.GlucoseLevel.ToString();
            row++;
        }

        row = 3;
        ExerciseDataTable exerciseDataTable = new ExerciseTableAdapter().FillByAll();
        for (int x = exerciseDataTable.Rows.Count - 1; x >= 0; x--)
        {
            range = workSheet.get_Range("E" + (row).ToString(), "F" + (row).ToString());
            range.Select();
            range.Font.Name = "Arial";
            range.HorizontalAlignment = Microsoft.Office.Interop.Excel.Constants.xlCenter;

            ExerciseDataRow exerciseDataRow = (ExerciseDataRow)exerciseDataTable.Rows[x];

            workSheet.Cells[row, 5] = exerciseDataRow.Date.Substring(0, 9);
            workSheet.Cells[row, 6] = exerciseDataRow.TotalSteps.ToString();
            row += 4;
        }

        excelApp.Visible = true;
    }

    #endregion

    private void textBoxGetsFocus(object sender, EventArgs e)
    {
        HideCaret(_textBox.Handle);
    }

    private void headPanelSizeChanged(object sender, EventArgs e)
    {
        if (_process != null)
        {
            MoveWindow(_process.MainWindowHandle, 0, 0, _headPanel.Width, _headPanel.Height, true);
        }
    }

    public Form getMainForm()
    {
        return _mainForm;
    }

    #region Delegates

    public delegate void continueAfterIntroScriptDelegate();
    public delegate void addContinueButtonDelegate();
    public delegate void setupPhysiciansNumberDelegate(string name);
    public delegate void setupPedometerReadingDelegate();
    public delegate void makeHeadTalkDelegate(string script, int groupID);
    public delegate void loadNextSceneDelegate();
    public delegate void loadAndPlayEndingVideoDelegate();
    public delegate void setupNamePromptDelegate(string name);

    #endregion

    #region DLLImport

    [System.Runtime.InteropServices.DllImport("user32.dll")]
    private static extern IntPtr SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

    [System.Runtime.InteropServices.DllImport("user32.dll")]
    private static extern bool MoveWindow(IntPtr hwnd, int x, int y, int cx, int cy, bool repaint);

    [System.Runtime.InteropServices.DllImport("user32.dll")]
    private static extern bool HideCaret(IntPtr hWnd);

    #endregion

    public static void Main()
    {
        new Graphics();
    }
}